/**
 * \file: sdc_daemon.h
 *
 * \version: $Id:$
 *
 * \brief : Definition of the interface used for communication of libSDC and the khd daemon
 * <b>SDC private header</b>
 *
 * \author: Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2014 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#ifndef __SDC_LIB_DAEMON_H_
#define __SDC_LIB_DAEMON_H_

#include <sdc.h>
#include <private/sdc_intern.h>
#include <sdc_perm.h>

/* Definitions types and defaults */

#define SDC_DAEMON_API __attribute__((__visibility__("default")))

/* msg length needs to be sufficient to store
 *  - sizeof(sdc_daemon_msg_header_type)
 *  - case install :
 *      sizeof(sdc_daemon_msg_install_type)
 *      max key length
 *          for RSA 4096:
 *              private key
 *                  header and version 7 bytes
 *                  8 integers up to 513 bytes + 4 bytes overhead
 *                      not all integers can be that long
 *                      anyway we take this value for computation
 *                  optional content ignored
 *                  4143 -> base64 encoding + pem header + linebreaks
 *                  6k will be more than sufficient
 *  - case remove :
 *      sizeof(sdc_daemon_msg_remove_type)
 */
#define SDC_DAEMON_MAX_MSG_LEN (6*1024) /**< Length in bytes */

#define SDC_CONFIG_DAEMON_SOCKET "listen_socket"

/** Installation operation flags */
typedef enum  {
    //FLAG_NONE     = 0,
    MSG_FLAG_VOLATILE       = (1 << 0),
    MSG_FLAG_RANDOM_KEY     = (1 << 1),

    MSG_FLAG_ALL        = (MSG_FLAG_VOLATILE | MSG_FLAG_RANDOM_KEY)
} sdc_daemon_msg_flags_type;

typedef enum {
    MSG_INSTALL_KEY,     /* Install operation type */
    MSG_IMPORT_KEY,      /* Key import */
    MSG_REMOVE_KEY       /* Remove operation type */
} sdc_daemon_msg_op_type;

typedef enum {
    MSG_SUCCESS,    /* Return type: success */
    MSG_ERROR       /* Return type: error */
} sdc_daemon_msg_result_type;

/** Header for request message sent from daemon to client */
typedef struct {
    size_t msg_len;                       /**< Length of whole message */
    sdc_daemon_msg_op_type operation;     /**< Type of operation */
} sdc_daemon_msg_header_type;

/** Operation specific message part for MSG_INSTALL_KEY operation */
typedef struct {
    sdc_daemon_msg_flags_type flags; /**< Installation flags */
    sdc_key_fmt_t fmt;              /**< requested key format */
    sdc_key_len_t len;              /**< requested key length */
    sdc_key_id_t req_id_min;        /**< min requested ID */
    sdc_key_id_t req_id_max;        /**< max requested ID - set \p req_id_min == \p req_id_max to request explicit id */
    sdc_permissions_t permissions;  /**< Permissions + group id of the key */
    size_t key_data_len;            /**< Length of key data appended after sdc_daemon_msg_install_type */
    sdc_keysecret_enc_t key_data_enc; /**< Encoding of key data appended after sdc_daemon_msg_install_type */
} sdc_daemon_msg_install_type;

typedef struct {
    sdc_daemon_msg_flags_type flags; /**< Import flags - only volatile is valid */
    sdc_key_fmt_t fmt;              /**< requested key format */
    sdc_key_len_t len;              /**< requested key length */
    sdc_key_id_t req_id_min;        /**< min requested ID */
    sdc_key_id_t req_id_max;        /**< max requested ID - set \p req_id_min == \p req_id_max to request explicit id */
    sdc_permissions_t permissions;  /**< Permissions + group id of the key */
    size_t key_data_len;            /**< Length of key data appended after sdc_daemon_msg_import_type */
    sdc_keysecret_enc_t key_data_enc; /**< Encoding of key data appended after sdc_daemon_msg_import_type */
} sdc_daemon_msg_import_type;

/** Operation specific message part for MSG_REMOVE_KEY operation */
typedef struct {
    sdc_key_id_t id;            /**< ID of key to remove */
} sdc_daemon_msg_remove_type;

/** Resulting data of an operation */
typedef union {
    sdc_key_id_t installed_id; /* in case of an successful MSG_INSTALL_KEY */
    sdc_error_t sdc_error;  /* in case of MSG_ERROR */
} sdc_daemon_msg_response_val_type;

/** Response message sent from daemon to client */
typedef struct {
    size_t msg_len;                     /**< Length of whole message */
    sdc_daemon_msg_result_type type;    /**< Return type */
    sdc_daemon_msg_response_val_type value; /**< Return type specific value */
} sdc_daemon_msg_response_type;

/* Functions */

SDC_DAEMON_API sdc_error_t sdc_daemon_check_permission_range(const sdc_permissions_t *perms);

#endif
